<?php
// includes/blocks/class-wc-paygenius-redirect-blocks.php

use Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType;

if (! defined('ABSPATH')) {
    exit;
}

class Paygenius_Hosted_Payment_Gateway_Redirect_Blocks extends AbstractPaymentMethodType
{

    /** @var Paygenius_Hosted_Payment_Gateway_Redirect */
    protected $gateway;

    /** Payment method id must match the gateway id */
    public function get_name()
    {
        return 'paygenius-redirect';
    }

    public function initialize()
    {
        // Load gateway to reuse its settings/title/description
        if (! class_exists('Paygenius_Hosted_Payment_Gateway_Redirect')) {
            require_once dirname(__DIR__, 1) . '/Paygenius_Hosted_Payment_Gateway_Redirect.php';
        }
        $this->gateway = new Paygenius_Hosted_Payment_Gateway_Redirect();
    }

    public function is_active()
    {
        if (! $this->gateway) {
            return false;
        }
        $enabled   = $this->gateway->get_option('enabled') === 'yes';
        $supported = method_exists($this->gateway, 'is_supported') ? $this->gateway->is_supported() : true;
        $activated = method_exists($this->gateway, 'is_activated') ? $this->gateway->is_activated() : true;
        return $enabled && $supported && $activated;
    }

    public function get_payment_method_script_handles() {
        $handle      = 'wc-paygenius-redirect-blocks';
        $script_rel  = 'assets/js/paygenius-blocks.js';
        $script_url  = PAYGENIUS_REDIRECT_URL . $script_rel;
        $script_path = PAYGENIUS_REDIRECT_DIR . $script_rel;

        wp_register_script(
            $handle,
            $script_url,
            array('wc-blocks-registry','wc-settings','wp-element','wp-i18n'),
            file_exists($script_path) ? filemtime($script_path) : '1.1.1',
            true
        );

        return array($handle);
    }



    public function get_payment_method_data()
    {
        return array(
            'title'                  => $this->gateway->title,
            'description'            => $this->gateway->description,
            'placeOrderButtonLabel'  => $this->gateway->order_button_text ?: __('Proceed to PayGenius', 'paygenius-hosted-payment-gateway'),
            // Feature flags for Blocks – minimal set for redirect gateways:
            'supports'               => array('features' => array('products')),
        );
    }

    /**
     * Blocks checkout calls this on the server to complete payment.
     * Return a redirect instruction for offsite checkout.
     *
     * @param \WC_Order $order
     * @return array
     * @throws \Exception on failure
     */
    public function process_payment($order)
    {
        if (! $order instanceof \WC_Order) {
            $order = wc_get_order($order);
        }
        if (! $order) {
            throw new \Exception(__('Invalid order.', 'paygenius-hosted-payment-gateway'));
        }

        if (! $this->gateway) {
            $this->initialize();
        }

        // Use the existing gateway logic to create the redirect session
        $redirect = $this->gateway->getRedirectUrl($order->get_id());

        if (! $redirect) {
            throw new \Exception(__('Unable to initiate PayGenius payment.', 'paygenius-hosted-payment-gateway'));
        }

        // Blocks expects a redirect response.
        return array(
            'type'         => 'redirect',
            'redirect_url' => esc_url_raw($redirect),
        );
    }
}
